// Escena de Three.js
const scene = new THREE.Scene();
const camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
renderer.toneMapping = THREE.ACESFilmicToneMapping;  // Mejora para HDRI
renderer.outputEncoding = THREE.sRGBEncoding;  // Mejor salida de color
document.body.appendChild(renderer.domElement);

// Iluminación
const ambientLight = new THREE.AmbientLight(0xffffff, 0.5); // Luz ambiental suave
scene.add(ambientLight);

// Luz direccional
const directionalLight = new THREE.DirectionalLight(0xffffff, 1);
directionalLight.position.set(5, 10, 7.5);
scene.add(directionalLight);

// Controles de órbita
const controls = new THREE.OrbitControls(camera, renderer.domElement);

// Cargador GLTF
const loader = new THREE.GLTFLoader();
let machineMesh;

// Cargar el modelo de la máquina
loader.load('./model/model5.glb', function (gltf) {
    machineMesh = gltf.scene;
    scene.add(machineMesh);

    // Ajustar las propiedades del material para usar reflejos
    machineMesh.traverse((child) => {
        if (child.isMesh) {
            child.material.metalness = 0.5;  // Valor inicial de metalicidad
            child.material.roughness = 0.5;  // Valor inicial de rugosidad
            child.material.envMap = scene.environment;  // Aplicar HDRI como reflejo
            child.material.needsUpdate = true;
        }
    });

    camera.position.z = 5; // Ajustar la cámara para ver el modelo
    controls.update();
}, undefined, function (error) {
    console.error('Error al cargar el modelo:', error);
});

// Cargar texturas
const textureLoader = new THREE.TextureLoader();

// Subir imágenes al modelo (4 áreas distintas)
function uploadImage(buttonId, inputId, meshName) {
    document.getElementById(buttonId).addEventListener('click', () => {
        document.getElementById(inputId).click();
    });

    document.getElementById(inputId).addEventListener('change', function (event) {
        const file = event.target.files[0];
        const reader = new FileReader();

        reader.onload = function (e) {
            const imageUrl = e.target.result;
            const texture = textureLoader.load(imageUrl);

            if (machineMesh) {
                machineMesh.traverse((child) => {
                    if (child.isMesh && child.name === meshName) {
                        child.material.map = texture;
                        child.material.needsUpdate = true;
                    }
                });
            }
        };

        reader.readAsDataURL(file); // Leer la imagen como DataURL
    });
}

// Configurar botones para subir imágenes
uploadImage('upload1', 'imageUpload1', 'BranArea');
uploadImage('upload2', 'imageUpload2', 'BrandArea2');
uploadImage('upload3', 'imageUpload3', 'BrandArea3');
uploadImage('upload4', 'imageUpload4', 'BrandArea4');

// Rotación y zoom del modelo
document.getElementById('rotateLeft').addEventListener('click', () => {
    machineMesh.rotation.y -= 0.1;
});

document.getElementById('rotateRight').addEventListener('click', () => {
    machineMesh.rotation.y += 0.1;
});

document.getElementById('zoomIn').addEventListener('click', () => {
    camera.position.z -= 0.5;
});

document.getElementById('zoomOut').addEventListener('click', () => {
    camera.position.z += 0.5;
});

// Seleccionar color de fondo
document.getElementById('bgColorPicker').addEventListener('input', function (event) {
    const color = event.target.value;
    scene.background = new THREE.Color(color);
});

// Cargar HDRI como fondo y reflejo
const rgbeLoader = new THREE.RGBELoader();
document.getElementById('hdriSelect').addEventListener('change', function (event) {
    const hdriPath = event.target.value;

    if (hdriPath) {
        rgbeLoader.load(hdriPath, function (texture) {
            texture.mapping = THREE.EquirectangularReflectionMapping;  // Mapeo HDRI
            scene.background = texture;  // Fondo HDRI
            scene.environment = texture; // Reflejos HDRI
        });
    } else {
        scene.background = null; // Sin fondo
        scene.environment = null; // Sin reflejos
    }
});

// Ajustar metalicidad y rugosidad
document.getElementById('metalnessRange').addEventListener('input', function (event) {
    const metalness = parseFloat(event.target.value);
    if (machineMesh) {
        machineMesh.traverse((child) => {
            if (child.isMesh) {
                child.material.metalness = metalness;
            }
        });
    }
});

document.getElementById('roughnessRange').addEventListener('input', function (event) {
    const roughness = parseFloat(event.target.value);
    if (machineMesh) {
        machineMesh.traverse((child) => {
            if (child.isMesh) {
                child.material.roughness = roughness;
            }
        });
    }
});

// Imprimir en PDF
document.getElementById('printPDF').addEventListener('click', () => {
    html2canvas(renderer.domElement).then(canvas => {
        const imgData = canvas.toDataURL('image/png');
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF();
        pdf.addImage(imgData, 'PNG', 10, 10, 180, 160);
        pdf.save('model_snapshot.pdf');
    });
});

// Animación
function animate() {
    requestAnimationFrame(animate);
    controls.update(); // Actualizar controles de órbita
    renderer.render(scene, camera);
}

animate();

// Manejar el cambio de tamaño de la ventana
window.addEventListener('resize', function () {
    camera.aspect = window.innerWidth / window.innerHeight;
    camera.updateProjectionMatrix();
    renderer.setSize(window.innerWidth, window.innerHeight);
});